;*** ADCapture2fl.asm ***;
;-------------------------------------------------
; Author: Torsten Knorr, create-soft@freenet.de
; Adapted to work with float array by PepeV.
;-------------------------------------------------
;*** Register Definitions ***;
 .def R_DATA_LOW   = R0
 .def R_DATA_HIGH  = R1
 .def R_SP_LOW     = R6
 .def R_SP_HIGH    = R7
 .def R_PARAM_LOW  = R10
 .def R_PARAM_HIGH = R11
 .def R_TEMP_LOW   = R22
 .def R_TEMP_HIGH  = R23
 .def R_COUNT_LOW  = R24
 .def R_COUNT_HIGH = R25
 .def R_XP_LOW     = R26
 .def R_XP_HIGH    = R27
 .def R_ZP_LOW     = R30
 .def R_ZP_HIGH    = R31
;-------------------------------------------------
;*** ADMUX - ADC Multiplexer Selection Register ***;
 .equ ADMUX = 0x07
 .equ MUX0  = 0    ; Analog Channel and Gain Selection Bits
 .equ MUX1  = 1    ; Analog Channel and Gain Selection Bits
 .equ MUX2  = 2    ; Analog Channel and Gain Selection Bits
 .equ MUX3  = 3    ; Analog Channel and Gain Selection Bits
 .equ MUX4  = 4    ; Analog Channel and Gain Selection Bits
 .equ MUXM  = 0x1F ; Analog Channel and Gain Selection Mask
 .equ ADLAR = 5	   ; Left Adjust Result
 .equ REFS0 = 6	   ; Reference Selection Bit 0
 .equ REFS1 = 7    ; Reference Selection Bit 1
 .equ REFSM = 0xC0 ; Reference Selection Mask
;-------------------------------------------------
;*** ADCSRA - ADC Control and Status Register A ***;
 .equ ADCSRA = 0x06
 .equ ADCSR  = ADCSRA ; For compatibility
 .equ ADPS0  = 0      ; ADC Prescaler Select Bits
 .equ ADPS1  = 1      ; ADC Prescaler Select Bits
 .equ ADPS2  = 2      ; ADC Prescaler Select Bits
 .equ ADPSM  = 0x07   ; ADC Prescaler Select Mask
 .equ ADIE   = 3      ; ADC Interrupt Enable
 .equ ADIF   = 4      ; ADC Interrupt Flag
 .equ ADFR   = 5      ; ADC  Free Running Select
 .equ ADSC   = 6      ; ADC Start Conversion
 .equ ADEN   = 7      ; ADC Enable
;-------------------------------------------------
;*** ADCH - ADC Data Register High Byte ***;
 .equ ADCH  = 0x05
;-------------------------------------------------
;*** ADCL - ADC Data Register Low Byte ***;
 .equ ADCL  = 0x04
;-------------------------------------------------
;*** SREG - Status Register ***;
 .equ SREG  = 0x3F
;-------------------------------------------------
;*** MCUCR - MCU Control Register ***;
 .equ MCUCR = 0x35
 .equ IVCE  = 0    ; Interrupt Vector Change Enable
 .equ IVSEL = 1    ; Interrupt Vector Select
 .equ SM2   = 2    ; Sleep Mode Select
 .equ SM0   = 3    ; Sleep Mode Select
 .equ SM1   = 4    ; Sleep Mode Select
 .equ SE    = 5    ; Sleep Enable
 .equ SRW10 = 6    ; External SRAM Wait State Select
 .equ SRE   = 7    ; External SRAM Enable
;-------------------------------------------------
 .ifdef TagADCaptureSetVref
 ADCaptureSetVref:

; get pointer to parameter
    movw R_ZP_LOW,    R_PARAM_LOW

; 1. parameter = by_vref
    ld   R_TEMP_LOW,  Z

; mask bits REFS0/REFS1
    andi R_TEMP_LOW,  REFSM

    cbi  ADMUX,       REFS0
    cbi  ADMUX,       REFS1
    in   R_TEMP_HIGH, ADMUX
    or   R_TEMP_LOW,  R_TEMP_HIGH

    out  ADMUX,       R_TEMP_LOW

    ret
 .endif
;-------------------------------------------------
 .ifdef TagADCaptureSetPrescaler
 ADCaptureSetPrescaler:

; get pointer to parameter
    movw R_ZP_LOW,    R_PARAM_LOW

; 1. parameter = by_prescaler
    ld   R_TEMP_LOW,  Z

; mask bits ADPS0/ADPS1/ADPS2
    andi R_TEMP_LOW,  ADPSM

    in   R_TEMP_HIGH, ADCSRA
    andi R_TEMP_HIGH, ~ADPSM
    or   R_TEMP_LOW,  R_TEMP_HIGH

    out  ADCSRA,      R_TEMP_LOW

    ret
 .endif
;-------------------------------------------------
 .ifdef TagADCaptureSetChannel
 ADCaptureSetChannel:

; get pointer to parameter
    movw R_ZP_LOW,    R_PARAM_LOW

; 1. parameter = by_channel
    ld   R_TEMP_LOW,  Z

; mask bits MUX0/MUX1/MUX2/MUX3/MUX4
    andi R_TEMP_LOW,  MUXM

    in   R_TEMP_HIGH, ADMUX
    andi R_TEMP_HIGH, ~MUXM
    or   R_TEMP_LOW,  R_TEMP_HIGH

    out  ADMUX,       R_TEMP_LOW

    ret
 .endif
;-------------------------------------------------
 .ifdef TagADCaptureRunFree2fl
 ADCaptureRunFree2fl:

    movw R_ZP_LOW,     R_PARAM_LOW              ; move parameter stack pointer into Z

    ld   R_COUNT_LOW,  Z+                       ; load 2nd parameter (w_buffer_size) into R_COUNT
    ld   R_COUNT_HIGH, Z+

; To keep track of the number of conversions,
; R_COUNT is decremented with each ADC until
; it reaches zero.

    ld   R_XP_LOW,     Z+                       ; load 1st parameter (p_buffer) into X
    ld   R_XP_HIGH,    Z

    sbi  ADCSRA,       ADFR                     ; set Free Running mode
    sbi  ADCSRA,       ADEN                     ; set ADC Enable
    sbi  ADCSRA,       ADSC                     ; set Start Conversion

 ADC_WAIT:                                      ; wait for ADC to complete
    sbis ADCSRA,       ADIF
    rjmp ADC_WAIT

    sbi  ADCSRA,       ADIF                     ; clear ADC interrupt flag

    in   R_TEMP_LOW,   ADCL                     ; move ADC value in R_TEMP
    in   R_TEMP_HIGH,  ADCH

; Conversion of 10 bit ADC value (in 16 bit word) to 32 bit float.

; The bits of the float from MSB to LSB are:
; 1 sign bit (m), 8 exponent bits (e),23 significant bits:
; meee eeee|esss ssss|ssss ssss|ssss ssss
; The exponent and the significant are found by repeatedly left shifting the
; ADC-value until the highest bit with value 1 is shifted in the carry (C=1).
; Then:
;    - the exponent is 16-n_shifts+127 = 143-n_shifts
;    - the significand is the left shifted ADC value padded right with zero's
;      to 23 bits
; The ADC value is positive so m=0.
; The highest value of a 16 bit (unsigned) integer is 1111 1111|1111 1111 so
; the highest value of the 23 bits significant is 111 1111|1111 1111|0000 0000.
; Therefore the last byte of the significant is always zero and any 16 bit
; integer can be represented as
;                 0eee eeee|esss ssss|ssss ssss|0000 0000

; ADC value 0000 0000|0000 0001 has exponent 0111 1111 and significant
; 000 0000|0000 0000|0000 0000.
; ADC value 0000 0000|0000 0000 has exponent 0000 0000 and significant
; 000 0000|0000 0000|0000 0000.
; This is an exeptional case that should be handled seperately by the code.
; If after 16 shifts left no carry is found, the ADC value is zero.

; The exponent is stored in R_ZP_LOW. The start value is 143 (no shifts).
; R_ZP_HIGH is used to limit the number of left shifts to 16. 
; R_TEMP contains the ADC value.
; A carry from left shift R_TEMP means the shifting is complete.

    ldi  R_ZP_LOW,     143                      ; set exponent to 143
    ldi  R_ZP_HIGH,    16                       ; maximum number of left shifts
    clr  R_DATA_HIGH                            ; set zero (constant)

 FIND_EXPONENT:
    dec  R_ZP_LOW                               ; count 1 shift
    lsl  R_TEMP_LOW                             ; left shift ADC value
    rol  R_TEMP_HIGH
    brcs EXPONENT_FOUND:                        ; if C=1, the left shifting is complete

    dec  R_ZP_HIGH                              ; if maximum of 16 left shifts not reached...
    brne FIND_EXPONENT                          ; ...continue shifting...
    rjmp STORE_VALUE                            ; ...else store ADC value (zero)

 EXPONENT_FOUND:
    lsr  R_TEMP_HIGH                            ; right shift ADC value
    ror  R_TEMP_LOW
    lsr  R_ZP_HIGH                              ; right shift exponenent
    brcc STORE_VALUE                            ; if C=1...
    sbr  R_TEMP_HIGH,  0x80                     ; ...set the MSB of the 2nd byte of the float

 STORE_VALUE:
    st   X+,           R_DATA_HIGH              ; 0th byte is zero
    st   X+,           R_TEMP_LOW               ; 1st byte
    st   X+,           R_TEMP_HIGH              ; 2nd byte
    st   X+,           R_ZP_LOW                 ; 3rd byte

; decrease counter
    sbiw R_COUNT_LOW, 1

; repeat ADC loop if R_COUNT is not zero
    brne ADC_WAIT

    ret
 .endif
;-------------------------------------------------
 .ifdef TagADCaptureRelease
 ADCaptureRelease:

    cbi ADCSRA, ADEN

    ret
 .endif
;-------------------------------------------------