;*** ADCapture2fl.asm ***;
;-------------------------------------------------
; Author: Torsten Knorr, create-soft@freenet.de
; Adapted to work with float array by PepeV.
;-------------------------------------------------
;*** Register Definitions ***;
 .def R_DATA_LOW   = R0
 .def R_DATA_HIGH  = R1
 .def R_SP_LOW     = R6
 .def R_SP_HIGH    = R7
 .def R_PARAM_LOW  = R10
 .def R_PARAM_HIGH = R11
 .def R_TEMP_LOW   = R22
 .def R_TEMP_HIGH  = R23
 .def R_COUNT_LOW  = R24
 .def R_COUNT_HIGH = R25
 .def R_XP_LOW     = R26
 .def R_XP_HIGH    = R27
 .def R_ZP_LOW     = R30
 .def R_ZP_HIGH    = R31
;-------------------------------------------------
;*** ADMUX - ADC Multiplexer Selection Register ***;
 .equ ADMUX = 0x07
 .equ MUX0  = 0    ; Analog Channel and Gain Selection Bits
 .equ MUX1  = 1    ; Analog Channel and Gain Selection Bits
 .equ MUX2  = 2    ; Analog Channel and Gain Selection Bits
 .equ MUX3  = 3    ; Analog Channel and Gain Selection Bits
 .equ MUX4  = 4    ; Analog Channel and Gain Selection Bits
 .equ MUXM  = 0x1F ; Analog Channel and Gain Selection Mask
 .equ ADLAR = 5	   ; Left Adjust Result
 .equ REFS0 = 6	   ; Reference Selection Bit 0
 .equ REFS1 = 7    ; Reference Selection Bit 1
 .equ REFSM = 0xC0 ; Reference Selection Mask
;-------------------------------------------------
;*** ADCSRA - ADC Control and Status Register A ***;
 .equ ADCSRA = 0x06
 .equ ADCSR  = ADCSRA ; For compatibility
 .equ ADPS0  = 0      ; ADC Prescaler Select Bits
 .equ ADPS1  = 1      ; ADC Prescaler Select Bits
 .equ ADPS2  = 2      ; ADC Prescaler Select Bits
 .equ ADPSM  = 0x07   ; ADC Prescaler Select Mask
 .equ ADIE   = 3      ; ADC Interrupt Enable
 .equ ADIF   = 4      ; ADC Interrupt Flag
 .equ ADFR   = 5      ; ADC  Free Running Select
 .equ ADSC   = 6      ; ADC Start Conversion
 .equ ADEN   = 7      ; ADC Enable
;-------------------------------------------------
;*** ADCH - ADC Data Register High Byte ***;
 .equ ADCH  = 0x05
;-------------------------------------------------
;*** ADCL - ADC Data Register Low Byte ***;
 .equ ADCL  = 0x04
;-------------------------------------------------
;*** SREG - Status Register ***;
 .equ SREG  = 0x3F
;-------------------------------------------------
;*** MCUCR - MCU Control Register ***;
 .equ MCUCR = 0x35
 .equ IVCE  = 0    ; Interrupt Vector Change Enable
 .equ IVSEL = 1    ; Interrupt Vector Select
 .equ SM2   = 2    ; Sleep Mode Select
 .equ SM0   = 3    ; Sleep Mode Select
 .equ SM1   = 4    ; Sleep Mode Select
 .equ SE    = 5    ; Sleep Enable
 .equ SRW10 = 6    ; External SRAM Wait State Select
 .equ SRE   = 7    ; External SRAM Enable
;-------------------------------------------------
 .ifdef TagADCaptureSetVref
 ADCaptureSetVref:

; get pointer to parameter
    movw R_ZP_LOW,    R_PARAM_LOW

; 1. parameter = by_vref
    ld   R_TEMP_LOW,  Z

; mask bits REFS0/REFS1
    andi R_TEMP_LOW,  REFSM

    cbi  ADMUX,       REFS0
    cbi  ADMUX,       REFS1
    in   R_TEMP_HIGH, ADMUX
    or   R_TEMP_LOW,  R_TEMP_HIGH

    out  ADMUX,       R_TEMP_LOW

    ret
 .endif
;-------------------------------------------------
 .ifdef TagADCaptureSetPrescaler
 ADCaptureSetPrescaler:

; get pointer to parameter
    movw R_ZP_LOW,    R_PARAM_LOW

; 1. parameter = by_prescaler
    ld   R_TEMP_LOW,  Z

; mask bits ADPS0/ADPS1/ADPS2
    andi R_TEMP_LOW,  ADPSM

    in   R_TEMP_HIGH, ADCSRA
    andi R_TEMP_HIGH, ~ADPSM
    or   R_TEMP_LOW,  R_TEMP_HIGH

    out  ADCSRA,      R_TEMP_LOW

    ret
 .endif
;-------------------------------------------------
 .ifdef TagADCaptureSetChannel
 ADCaptureSetChannel:

; get pointer to parameter
    movw R_ZP_LOW,    R_PARAM_LOW

; 1. parameter = by_channel
    ld   R_TEMP_LOW,  Z

; mask bits MUX0/MUX1/MUX2/MUX3/MUX4
    andi R_TEMP_LOW,  MUXM

    in   R_TEMP_HIGH, ADMUX
    andi R_TEMP_HIGH, ~MUXM
    or   R_TEMP_LOW,  R_TEMP_HIGH

    out  ADMUX,       R_TEMP_LOW

    ret
 .endif
;-------------------------------------------------
 .ifdef TagADCaptureRunFree2fl
 ADCaptureRunFree2fl:

    movw R_XP_LOW,     R_PARAM_LOW              ; move parameter stack pointer into X

    ld   R_COUNT_LOW,  X+                       ; load 2nd parameter (w_buffer_size) into R_COUNT
    ld   R_COUNT_HIGH, X+

; To keep track of the number of conversions,
; R_COUNT is decremented with each ADC until
; it reaches zero.

    ld   R_ZP_LOW,     X+                       ; load 1st parameter (p_buffer) into Z
    ld   R_ZP_HIGH,    X

    sbi  ADCSRA,       ADFR                     ; set Free Running mode
    sbi  ADCSRA,       ADEN                     ; set ADC Enable
    sbi  ADCSRA,       ADSC                     ; set Start Conversion

 ADC_WAIT:                                      ; wait for ADC to complete
    sbis ADCSRA,       ADIF
    rjmp ADC_WAIT

    sbi  ADCSRA,       ADIF                     ; clear ADC interrupt flag

    in   R_TEMP_LOW,   ADCL                     ; move ADC value in R_TEMP
    in   R_TEMP_HIGH,  ADCH

; Conversion of 10 bit ADC value (in 16 bit word) to 32 bit float.

; The bits of the float from MSB to LSB are:
; 1 sign bit (m), 8 exponent bits (e),23 significant bits:
; meee eeee|esss ssss|ssss ssss|ssss ssss
; The exponent and the significant are found by repeatedly left shifting the
; ADC-value until the highest bit with value 1 is shifted in the carry (C=1).
; Then:
;    - the exponent is 16-n_shifts+127 = 143-n_shifts
;    - the significand is the left shifted ADC value padded right with zero's
;      to 23 bits
; The ADC value is positive so m=0.
; The highest ADC value is 0000 0011|1111 1111 so the highest value of the
; 23 bits significant is 111 1111|1100 0000|0000 0000.
; Therefore the last byte of the significant is always zero and any ADC value
; can be represented as
;                 0eee eeee|esss ssss|ss00 0000|0000 0000

; ADC value 0000 0000|0000 0001 has exponent 0111 1111 and significant
; 000 0000|0000 0000|0000 0000.
; ADC value 0000 0000|0000 0000 has exponent 0000 0000 and significant
; 000 0000|0000 0000|0000 0000.
; This is an exeptional case that should be handled seperately by the code
; as the left shift algoritm above would continue forever causing the program
; to hang.
; The 4 bytes of the float are filled as follows (high byte to low byte):
; byte 3 = exponent>>1
; byte 2 : byte 1 = significand>>1 with MSB of byte 2 equal to LSB of exponent
; byte 0 = 0000 0000
; The result is stored in 4 bytes of the buffer, counting from from the current
; value of the Z pointer

; 1. Check if ADC value is zero.
    mov  R_XP_LOW,     R_TEMP_LOW
    or   R_XP_LOW,     R_TEMP_HIGH
    brne START_SHIFTING                         ; if Z=0, ADC value is not zero
;   Z=1, ADC value is zero, therefore all bytes of float are zero.
    clr  R_XP_LOW                               ; clear R_XP_LOW
    std  Z+3,          R_XP_LOW                 ; store 0 in byte 3
    std  Z+2,          R_XP_LOW                 ; store 0 in byte 2
    std  Z+1,          R_XP_LOW                 ; store 0 in byte 1
    st   Z,            R_XP_LOW                 ; store 0 in byte 0
    rjmp NEXT_ELEMENT

; 2. Repeat left shift ADC value until C is one.
;    The exponent is stored in R_XP_LOW. The start value is 143 (no shifts).
;    R_TEMP contains the significant.
;    A carry from left shift R_TEMP_HIGH means the shifting is complete.
;    This is marked by setting T.

 START_SHIFTING:
    ldi  R_XP_LOW,     143                      ; set exponent to 143
    clt                                         ; clear T

 SHIFT_LEFT_ADC_VALUE:                          ; begin of loop
    lsl  R_TEMP_LOW                             ; left shift ADC value
    rol  R_TEMP_HIGH
    brcc DEC_EXPONENT                           ; if C=0 skip next instruction
    set                                         ; set T=1: shifting complete

 DEC_EXPONENT:
    dec  R_XP_LOW                               ; decrease exponent with 1
    brtc SHIFT_LEFT_ADC_VALUE                   ; if T=0 continue shifting

; 3. Store the bytes of the calculated float
;   Calculate byte 3
    bst  R_XP_LOW,     0                        ; save bit 0 of exponent in T
    lsr  R_XP_LOW                               ; byte 3 is exponent>>1
    std  Z+3,          R_XP_LOW                 ; store byte 3

;   Calculate bytes 2 and 1
    lsr  R_TEMP_HIGH                            ; rotate right significant
    ror  R_TEMP_LOW
    bld  R_TEMP_HIGH,  7                        ; put LSB(exponent) in MSB(byte 2)
    std  Z+2,          R_TEMP_HIGH              ; store byte 2
    std  Z+1,          R_TEMP_LOW               ; store byte 1

;   Calculate byte 0
    clr  R_XP_LOW                               ; byte 0 is 0
    st   Z,            R_XP_LOW                 ; store byte 0

 NEXT_ELEMENT:
; make Z point to the next element of the buffer (+4 bytes)
    adiw R_ZP_LOW,    4

; decrease counter
    sbiw R_COUNT_LOW, 1

; repeat ADC loop if R_COUNT is not zero
    brne ADC_WAIT

    ret
 .endif
;-------------------------------------------------
 .ifdef TagADCaptureRelease
 ADCaptureRelease:

    cbi ADCSRA, ADEN

    ret
 .endif
;-------------------------------------------------